/**
 * 运动控制API工具模块
 * Motion Control API Utility Module
 */

class MotionControlApi {
  constructor() {
    this.baseUrl = '';
    this.isInitialized = false;
  }

  /**
   * 初始化API，从存储中获取服务器地址
   */
  init() {
    try {
      const serverUrl = wx.getStorageSync('serverUrl');
      if (serverUrl) {
        this.baseUrl = serverUrl.startsWith('http') ? serverUrl : `http://${serverUrl}`;
        this.isInitialized = true;
        console.log('Motion Control API initialized:', this.baseUrl);
      } else {
        console.warn('Server URL not configured');
        this.isInitialized = false;
      }
    } catch (error) {
      console.error('Failed to initialize Motion Control API:', error);
      this.isInitialized = false;
    }
  }

  /**
   * 检查API是否已初始化
   */
  checkInit() {
    if (!this.isInitialized) {
      wx.showToast({
        icon: 'error',
        title: '请先在设置中初始化'
      });
      return false;
    }
    return true;
  }

  /**
   * 通用API请求方法
   */
  async request(endpoint, data) {
    if (!this.isInitialized) {
      // 不要显示toast，因为调用方已经处理了提示
      console.warn('API not initialized, skipping request:', endpoint);
      return Promise.reject({ message: 'API not initialized' });
    }

    const url = `${this.baseUrl}${endpoint}`;
    
    try {
      console.log(`API Request: ${url}`, data);
      
      const response = await new Promise((resolve, reject) => {
        wx.request({
          url: url,
          method: 'POST',
          data: data,
          header: {
            'Content-Type': 'application/json'
          },
          success: resolve,
          fail: reject
        });
      });

      if (response.statusCode === 200) {
        console.log('API Response:', response.data);
        return response.data;
      } else {
        throw new Error(`HTTP ${response.statusCode}: ${response.data?.message || 'Unknown error'}`);
      }
    } catch (error) {
      console.error(`API Error [${endpoint}]:`, error);
      this.handleApiError(error);
      throw error;
    }
  }

  /**
   * 处理API错误
   */
  handleApiError(error) {
    let title = '操作失败';
    
    if (error.statusCode) {
      switch (error.statusCode) {
        case 401:
          title = '命令参数缺失';
          break;
        case 402:
          title = '命令参数非法';
          break;
        case 403:
          title = '移动距离无效';
          break;
        case 405:
          title = '移动模式非法';
          break;
        case 429:
          title = '请求过于频繁';
          break;
        case 500:
          title = '服务器内部错误';
          break;
        default:
          title = `错误 ${error.statusCode}`;
      }
    } else if (error.errMsg) {
      if (error.errMsg.includes('timeout')) {
        title = '连接超时';
      } else if (error.errMsg.includes('fail')) {
        title = '连接失败';
      }
    }

    wx.showToast({
      icon: 'error',
      title: title,
      duration: 2000
    });
  }

  /**
   * 1. 控制设备状态 (启动、暂停、复位、停止、继续)
   */
  async controlMachine(deviceCommand, deviceMode = true) {
    const endpoint = '/api/MachineControl/ControlMachineStatus';
    const data = {
      deviceMode: deviceMode,
      deviceCommand: deviceCommand
    };
    return await this.request(endpoint, data);
  }

  /**
   * 2. 控制轴寸动
   */
  async controlAxisInchMove(axisName, moveMode, inchDistance, isRapid = false) {
    const endpoint = '/api/MachineControl/ControlAxisInchMove';
    const data = {
      axisName: axisName,
      moveMode: moveMode,
      inchDistance: inchDistance,
      isRapid: isRapid
    };
    return await this.request(endpoint, data);
  }

  /**
   * 3. 控制轴启停
   */
  async controlAxis(axisName, moveMode, isRapid = false) {
    const endpoint = '/api/MachineControl/ControlAxis';
    const data = {
      axisName: axisName,
      moveMode: moveMode,
      isRapid: isRapid
    };
    return await this.request(endpoint, data);
  }

  /**
   * 4. 控制驱动器使能
   */
  async enableDriver(isEnable) {
    const endpoint = '/api/MachineControl/EnableDriver';
    const data = {
      isEnable: isEnable
    };
    return await this.request(endpoint, data);
  }

  /**
   * 5. 控制轴回原点
   */
  async backZeroPoint(axisName) {
    const endpoint = '/api/MachineControl/BackZeroPoint';
    const data = {
      axisName: axisName
    };
    return await this.request(endpoint, data);
  }

  // 便捷方法：设备控制
  async startMachine() {
    return await this.controlMachine(1); // 启动
  }

  async pauseMachine() {
    return await this.controlMachine(2); // 暂停
  }

  async resetMachine() {
    return await this.controlMachine(3); // 复位
  }

  async stopMachine() {
    return await this.controlMachine(4); // 停止
  }

  async continueMachine() {
    return await this.controlMachine(5); // 继续
  }

  // 便捷方法：轴控制
  async moveAxisLeft(isRapid = false) {
    return await this.controlAxis(0, 0, isRapid); // X轴正向
  }

  async moveAxisRight(isRapid = false) {
    return await this.controlAxis(0, 1, isRapid); // X轴负向
  }

  async moveAxisUp(isRapid = false) {
    return await this.controlAxis(1, 0, isRapid); // Y轴正向
  }

  async moveAxisDown(isRapid = false) {
    return await this.controlAxis(1, 1, isRapid); // Y轴负向
  }

  async moveZAxisUp(isRapid = false) {
    return await this.controlAxis(2, 0, isRapid); // Z轴上
  }

  async moveZAxisDown(isRapid = false) {
    return await this.controlAxis(2, 1, isRapid); // Z轴下
  }

  async stopAxis(axisName) {
    return await this.controlAxis(axisName, 2, false); // 停止轴
  }

  async stopAllAxes() {
    // 停止所有轴
    const promises = [
      this.stopAxis(0), // X轴
      this.stopAxis(1), // Y轴
      this.stopAxis(2), // Z轴
      this.stopAxis(3)  // F轴
    ];
    return await Promise.allSettled(promises);
  }

  // 便捷方法：回原点
  async backToOriginX() {
    return await this.backZeroPoint(0); // X轴回原点
  }

  async backToOriginY() {
    return await this.backZeroPoint(1); // Y轴回原点
  }

  async backToOriginZ() {
    return await this.backZeroPoint(2); // Z轴回原点
  }

  async backToOriginF() {
    return await this.backZeroPoint(3); // F轴回原点
  }

  async backToOriginAll() {
    return await this.backZeroPoint(4); // 所有轴回原点
  }
}

// 创建单例实例
const motionControlApi = new MotionControlApi();

module.exports = motionControlApi; 