/**
 * 配置管理器 - Configuration Manager
 * 统一管理应用配置，支持热加载和自定义配置
 */

const AppConfig = require('../config/app-config')

class ConfigManager {
  constructor() {
    this.config = AppConfig
    this.customConfig = {}
    this.isInitialized = false
  }

  /**
   * 初始化配置管理器
   */
  init() {
    try {
      // 加载自定义配置
      this.loadCustomConfig()
      
      // 合并配置
      this.mergeConfigs()
      
      // 验证配置完整性
      if (!this.validateConfig()) {
        console.warn('Configuration validation failed, using default config')
        this.config = AppConfig
      }
      
      this.isInitialized = true
      console.log('Configuration Manager initialized successfully')
      return true
    } catch (error) {
      console.error('Failed to initialize Configuration Manager:', error)
      // 使用默认配置作为备用
      this.config = AppConfig
      this.isInitialized = true
      console.log('Using default configuration as fallback')
      return false
    }
  }

  /**
   * 加载自定义配置
   */
  loadCustomConfig() {
    try {
      const customConfigStr = wx.getStorageSync('customConfig')
      if (customConfigStr) {
        this.customConfig = JSON.parse(customConfigStr)
        console.log('Custom configuration loaded')
      }
    } catch (error) {
      console.warn('Failed to load custom configuration:', error)
      this.customConfig = {}
    }
  }

  /**
   * 保存自定义配置
   */
  saveCustomConfig(config) {
    try {
      this.customConfig = { ...this.customConfig, ...config }
      wx.setStorageSync('customConfig', JSON.stringify(this.customConfig))
      this.mergeConfigs()
      console.log('Custom configuration saved')
      return true
    } catch (error) {
      console.error('Failed to save custom configuration:', error)
      return false
    }
  }

  /**
   * 合并配置
   */
  mergeConfigs() {
    this.config = this.deepMerge(AppConfig, this.customConfig)
  }

  /**
   * 深度合并对象
   */
  deepMerge(target, source) {
    const result = { ...target }
    
    for (const key in source) {
      if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
        result[key] = this.deepMerge(target[key] || {}, source[key])
      } else {
        result[key] = source[key]
      }
    }
    
    return result
  }

  /**
   * 获取配置值
   */
  get(path, defaultValue = null) {
    try {
      const keys = path.split('.')
      let value = this.config
      
      for (const key of keys) {
        if (value && typeof value === 'object' && key in value) {
          value = value[key]
        } else {
          return defaultValue
        }
      }
      
      return value
    } catch (error) {
      console.error('Failed to get config value:', error)
      return defaultValue
    }
  }

  /**
   * 设置配置值
   */
  set(path, value) {
    try {
      const keys = path.split('.')
      const lastKey = keys.pop()
      let target = this.config
      
      for (const key of keys) {
        if (!target[key] || typeof target[key] !== 'object') {
          target[key] = {}
        }
        target = target[key]
      }
      
      target[lastKey] = value
      
      // 同步到自定义配置
      const customPath = {}
      let customTarget = customPath
      for (const key of keys) {
        customTarget[key] = {}
        customTarget = customTarget[key]
      }
      customTarget[lastKey] = value
      
      this.saveCustomConfig(customPath)
      return true
    } catch (error) {
      console.error('Failed to set config value:', error)
      return false
    }
  }

  /**
   * 获取应用信息
   */
  getAppInfo() {
    return this.get('app', {})
  }

  /**
   * 获取主题配置
   */
  getTheme() {
    return this.get('theme', {})
  }

  /**
   * 获取当前主题配置
   */
  getCurrentTheme() {
    const theme = this.getTheme()
    const currentThemeName = theme.currentTheme || 'default'
    const themes = theme.themes || {}
    
    // 返回当前选中的主题配置
    return themes[currentThemeName] || themes.default || {
      name: '默认主题',
      background: 'linear-gradient(135deg, #323844 0%, #2c3e50 100%)',
      cardBackground: 'linear-gradient(145deg, rgba(255,255,255,0.15), rgba(255,255,255,0.08))',
      textPrimary: '#ffffff',
      textSecondary: '#bdc3c7',
      accent: '#3498db',
      navbarBackground: 'rgba(0, 0, 0, 0.5)',
      overlayBackground: 'rgba(128, 128, 128, 0.3)'
    }
  }

  /**
   * 获取可用主题列表
   */
  getAvailableThemes() {
    const theme = this.getTheme()
    const themes = theme.themes || {}
    
    return Object.keys(themes).map(key => ({
      id: key,
      name: themes[key].name || key,
      config: themes[key]
    }))
  }

  /**
   * 切换主题
   */
  switchTheme(themeName) {
    try {
      const theme = this.getTheme()
      const themes = theme.themes || {}
      
      if (!themes[themeName]) {
        console.warn(`Theme '${themeName}' not found, falling back to default`)
        themeName = 'default'
      }
      
      // 更新当前主题选择
      this.set('theme.currentTheme', themeName)
      
      console.log(`Theme switched to: ${themeName}`)
      return true
    } catch (error) {
      console.error('Failed to switch theme:', error)
      return false
    }
  }

  /**
   * 获取当前主题名称
   */
  getCurrentThemeName() {
    return this.get('theme.currentTheme', 'default')
  }

  /**
   * 获取按钮布局
   */
  getButtonLayout() {
    return this.get('buttonLayout', {})
  }

  /**
   * 获取API配置
   */
  getApiConfig() {
    return this.get('api', {})
  }

  /**
   * 获取动作映射
   */
  getActionMappings() {
    return this.get('actionMappings', {})
  }

  /**
   * 获取侧边菜单配置
   */
  getSideMenuConfig() {
    return this.get('sideMenu', {})
  }

  /**
   * 获取功能开关
   */
  getFeatures() {
    return this.get('features', {})
  }

  /**
   * 获取默认设置
   */
  getDefaults() {
    return this.get('defaults', {})
  }

  /**
   * 应用主题到页面
   */
  applyTheme(pageInstance) {
    try {
      const theme = this.getTheme()
      const currentTheme = this.getCurrentTheme()
      
      // 设置主题数据到页面
      pageInstance.setData({
        themeConfig: theme,
        currentTheme: currentTheme,
        currentThemeName: this.getCurrentThemeName(),
        availableThemes: this.getAvailableThemes(),
        primaryColors: currentTheme, // 使用当前主题的配置
        buttonColors: theme.buttonColors,
        backgroundImage: theme.backgroundImage,
        // 添加主题相关的样式变量
        themeBackground: currentTheme.background,
        themeCardBackground: currentTheme.cardBackground,
        themeTextPrimary: currentTheme.textPrimary,
        themeTextSecondary: currentTheme.textSecondary,
        themeAccent: currentTheme.accent,
        themeNavbarBackground: currentTheme.navbarBackground,
        themeOverlayBackground: currentTheme.overlayBackground
      })
      
      console.log(`Theme '${this.getCurrentThemeName()}' applied to page`)
    } catch (error) {
      console.error('Failed to apply theme:', error)
    }
  }

  /**
   * 应用主题到指定页面实例（支持动态切换）
   */
  applyThemeToPage(pageInstance, themeName = null) {
    try {
      // 如果指定了主题名称，先切换主题
      if (themeName && themeName !== this.getCurrentThemeName()) {
        this.switchTheme(themeName)
      }
      
      // 应用主题
      this.applyTheme(pageInstance)
      
      return true
    } catch (error) {
      console.error('Failed to apply theme to page:', error)
      return false
    }
  }

  /**
   * 获取按钮配置
   */
  getButtonConfig(index) {
    const buttons = this.get('buttonLayout.buttons', [])
    return buttons[index] || null
  }

  /**
   * 获取动作配置
   */
  getActionConfig(actionName) {
    const actionMappings = this.getActionMappings()
    return actionMappings[actionName] || null
  }

  /**
   * 检查功能是否启用
   */
  isFeatureEnabled(featureName) {
    const features = this.getFeatures()
    return features[featureName] === true
  }

  /**
   * 重置配置到默认值
   */
  resetToDefaults() {
    try {
      this.customConfig = {}
      wx.removeStorageSync('customConfig')
      this.mergeConfigs()
      console.log('Configuration reset to defaults')
      return true
    } catch (error) {
      console.error('Failed to reset configuration:', error)
      return false
    }
  }

  /**
   * 导出配置
   */
  exportConfig() {
    try {
      return JSON.stringify(this.config, null, 2)
    } catch (error) {
      console.error('Failed to export configuration:', error)
      return null
    }
  }

  /**
   * 导入配置
   */
  importConfig(configStr) {
    try {
      const importedConfig = JSON.parse(configStr)
      this.saveCustomConfig(importedConfig)
      console.log('Configuration imported successfully')
      return true
    } catch (error) {
      console.error('Failed to import configuration:', error)
      return false
    }
  }

  /**
   * 验证配置
   */
  validateConfig(config = this.config) {
    const required = ['app', 'theme', 'buttonLayout', 'api', 'actionMappings']
    
    for (const key of required) {
      if (!config[key]) {
        console.error(`Missing required configuration: ${key}`)
        return false
      }
    }
    
    return true
  }
}

// 创建单例实例
const configManager = new ConfigManager()

module.exports = configManager 