// index.js
// 获取应用实例
const app = getApp()
// 引入运动控制API
const motionControlApi = require('../../utils/motionControlApi')
// 引入配置管理器
const configManager = require('../../utils/configManager')

Page({
  data: {
    sideMenu: false,
    linkState: false,
    keys: '', // 发送的消息的key
    isRapidMode: false, // 高速模式标志
    stepDistance: 1.0, // 步进距离 (mm)
    sideMenuList: [],
    buttons: [],
    hideKeyboard: false, // 键盘隐藏标志
    // 主题相关
    themeConfig: {},
    primaryColors: {},
    buttonColors: {},
    backgroundImage: '',
    // 应用信息
    appName: '虚拟键盘',
    appVersion: 'v1.0.0',
    logoPath: '/static/logo.png',
    statusBarHeight: 44 // Added for status bar height handling
  },

  onLoad() {
    console.log('=== onLoad started ===')
    
    // 获取系统信息，设置状态栏高度
    try {
      const systemInfo = wx.getSystemInfoSync()
      const statusBarHeight = systemInfo.statusBarHeight || 44
      
      // 动态设置CSS变量
      if (typeof document !== 'undefined') {
        document.documentElement.style.setProperty('--status-bar-height', `${statusBarHeight}px`)
      } else {
        // 小程序环境下的替代方案
        this.setData({
          statusBarHeight: statusBarHeight
        })
      }
      
      console.log('Status bar height set to:', statusBarHeight)
    } catch (error) {
      console.warn('Failed to get system info, using default status bar height:', error)
    }
    
    // 直接设置简单的按钮数据用于测试
    const testButtons = [
      { text: '开始', type: 'green', action: 'start' },
      { text: '暂停', type: 'brown', action: 'pause' },
      { text: '继续', type: 'blue', action: 'continue' },
      { text: '停止', type: 'red', action: 'stop' },
      { text: '吹气', type: 'white', action: 'blow' },
      { text: '跟随', type: 'white', action: 'follow' },
      { text: '清除\n报警', type: 'white', action: 'clearAlarm' },
      { text: '红光', type: 'white', action: 'redLight' },
      { text: '回退', type: 'white', action: 'backward' },
      { text: '前进', type: 'white', action: 'forward' },
      { text: 'K1', type: 'blue', action: 'k1' },
      { text: 'K2', type: 'blue', action: 'k2' },
      { text: '寻边', type: 'blue', action: 'findEdge' },
      { arrow: true, type: 'white', rotation: 0, action: 'moveUp' },
      { text: '标定', type: 'gray', action: 'calibrate' },
      { text: 'Fn', type: 'blue', action: 'fn' },
      { arrow: true, type: 'white', rotation: -90, action: 'moveLeft' },
      { text: '工作\n零点', type: 'blue', action: 'workZero' },
      { arrow: true, type: 'white', rotation: 90, action: 'moveRight' },
      { text: '高速', type: 'yellow', action: 'highSpeed' },
      { text: '走边框', type: 'blue', action: 'walkBorder' },
      { arrow: true, type: 'white', rotation: 180, action: 'moveDown' },
      { zAxis: true, type: 'gray', rotation: 0, action: 'zUp' },
      { text: '步进', type: 'gray', action: 'step' }
    ]
    
    const testSideMenu = [
      { text: '设置', icon: '⚙️', url: '../setting/setting' },
      { text: '关于', icon: 'ℹ️', url: '../about/about' }
    ]
    
    this.setData({
      buttons: testButtons,
      sideMenuList: testSideMenu,
      keys: ''.padStart(32, '0'),
      isRapidMode: false,
      stepDistance: 1.0,
      linkState: false,
      sideMenu: false,
      // 添加应用信息确保logo显示
      appName: '虚拟键盘',
      appVersion: 'v1.0.0',
      logoPath: '/static/logo.png'
    })
    
    console.log('Test buttons set - length:', testButtons.length)
    console.log('Button data sample:', testButtons.slice(0, 3))
    
    // 尝试加载配置（不影响基本功能）
    try {
      this.loadConfiguration()
    } catch (error) {
      console.warn('Configuration loading failed, using test data:', error)
    }
    
    this.initializeApi()
    this.setTimer()
    
    console.log('=== onLoad completed ===')
  },

  /**
   * 加载配置
   */
  loadConfiguration() {
    console.log('Starting configuration loading...')
    
    try {
      // 先设置默认配置确保界面可用
      console.log('Setting default configuration first...')
      this.loadDefaultConfiguration()
      
      // 然后尝试加载自定义配置
      console.log('Attempting to load custom configuration...')
      const configInitialized = configManager.init()
      
      if (configInitialized) {
        console.log('Configuration manager initialized successfully')
        
        // 加载按钮布局
        const buttonLayout = configManager.getButtonLayout()
        console.log('Button layout loaded:', buttonLayout)
        
        // 加载侧边菜单
        const sideMenuConfig = configManager.getSideMenuConfig()
        console.log('Side menu config loaded:', sideMenuConfig)
        
        // 如果配置有效，使用配置的按钮
        if (buttonLayout && buttonLayout.buttons && buttonLayout.buttons.length > 0) {
          console.log('Using configuration buttons:', buttonLayout.buttons.length)
          this.setData({
            buttons: buttonLayout.buttons
          })
        }
        
        // 如果侧边菜单配置有效，使用配置的菜单
        if (sideMenuConfig && sideMenuConfig.items && sideMenuConfig.items.length > 0) {
          console.log('Using configuration side menu:', sideMenuConfig.items.length)
          this.setData({
            sideMenuList: sideMenuConfig.items
          })
        }
        
        // 应用主题
        configManager.applyTheme(this)
        
        // 加载应用信息
        const appInfo = configManager.getAppInfo()
        if (appInfo) {
          this.setData({
            appName: appInfo.name || '虚拟键盘',
            appVersion: appInfo.version || 'v1.0.0',
            logoPath: appInfo.logo || '/static/logo.png'
          })
        }
        
        // 加载其他设置
        const defaults = configManager.getDefaults()
        if (defaults) {
          this.setData({
            isRapidMode: defaults.isRapidMode || false,
            stepDistance: wx.getStorageSync('stepDistance') || defaults.stepDistance || 1.0
          })
        }
        
        console.log('Configuration loaded successfully')
      } else {
        console.warn('Configuration manager failed to initialize, using default configuration')
      }
      
    } catch (error) {
      console.error('Failed to load configuration:', error)
      // 确保有默认配置
      this.loadDefaultConfiguration()
    }
    
    // 最终检查确保有按钮
    if (!this.data.buttons || this.data.buttons.length === 0) {
      console.warn('No buttons found, forcing default buttons')
      this.setData({
        buttons: this.getDefaultButtons()
      })
    }
    
    console.log('Final button count:', this.data.buttons ? this.data.buttons.length : 0)
  },

  /**
   * 获取默认按钮配置
   */
  getDefaultButtons() {
    return [
      // 第一行 - 设备控制
      { text: '开始', type: 'green', action: 'start', icon: '▶️' },
      { text: '暂停', type: 'brown', action: 'pause', icon: '⏸️' },
      { text: '继续', type: 'blue', action: 'continue', icon: '⏯️' },
      { text: '停止', type: 'red', action: 'stop', icon: '⏹️' },
      { text: '吹气', type: 'white', action: 'blow', icon: '💨' },
      { text: '跟随', type: 'white', action: 'follow', icon: '👥' },
      
      // 第二行 - 功能控制
      { text: '清除\n报警', type: 'white', action: 'clearAlarm', icon: '🚨' },
      { text: '红光', type: 'white', action: 'redLight', icon: '🔴' },
      { text: '断点\n定位', type: 'white', action: 'breakpointPos', icon: '📍' },
      { text: '回退', type: 'white', action: 'backward', icon: '⏪' },
      { text: '前进', type: 'white', action: 'forward', icon: '⏩' },
      { text: '激光\n点射', type: 'yellow', action: 'laserPulse', icon: '⚡' },
      
      // 第三行 - 方向控制（上）
      { text: 'K1', type: 'blue', action: 'k1', icon: '1️⃣' },
      { text: 'K2', type: 'blue', action: 'k2', icon: '2️⃣' },
      { text: 'K3', type: 'blue', action: 'k3', icon: '3️⃣' },
      { text: 'K4', type: 'blue', action: 'k4', icon: '4️⃣' },
      { text: '寻边', type: 'blue', showFn: true, action: 'findEdge', icon: '🔍' },
      { arrow: true, type: 'white', rotation: 0, action: 'moveUp', icon: '⬆️' },
      
      // 第四行 - 方向控制（中）
      { text: '一键\n交换', type: 'blue', showFn: true, action: 'quickSwap', icon: '🔄' },
      { text: '标定', type: 'gray', action: 'calibrate', icon: '📐' },
      { text: 'Fn', type: 'blue', action: 'fn', icon: '⚙️' },
      { text: '空运行', type: 'blue', showFn: true, action: 'dryRun', icon: '🏃' },
      { arrow: true, type: 'white', rotation: -90, action: 'moveLeft', icon: '⬅️' },
      { text: '工作\n零点', type: 'blue', showFn: true, action: 'workZero', icon: '🎯' },
      
      // 第五行 - 方向控制（下）
      { arrow: true, type: 'white', rotation: 90, action: 'moveRight', icon: '➡️' },
      { text: '高速', type: 'yellow', action: 'highSpeed', icon: '🚀' },
      { text: '走边框', type: 'blue', showFn: true, action: 'walkBorder', icon: '🔲' },
      { arrow: true, type: 'white', rotation: 180, action: 'moveDown', icon: '⬇️' },
      { zAxis: true, type: 'gray', rotation: 0, action: 'zUp', icon: '🔼' },
      { zAxis: true, type: 'gray', rotation: 180, action: 'zDown', icon: '🔽' },
      
      // 第六行 - 扩展功能
      { text: '低速', type: 'gray', action: 'lowSpeed', icon: '🐌' },
      { text: '步进', type: 'gray', action: 'step', icon: '📏' },
      { text: '', type: 'gray' }, // 空白按钮
      { text: '', type: 'gray' }, // 空白按钮
      { text: '', type: 'gray' }, // 空白按钮
      { text: '', type: 'gray' }  // 空白按钮
    ]
  },

  /**
   * 获取默认侧边菜单
   */
  getDefaultSideMenu() {
    return [
      { text: '设置', icon: '⚙️', url: '../setting/setting' },
      { text: '关于', icon: 'ℹ️', url: '../about/about' }
    ]
  },

  /**
   * 加载默认配置（备用）
   */
  loadDefaultConfiguration() {
    console.log('Loading default configuration...')
    
    const defaultButtons = this.getDefaultButtons()
    const defaultSideMenu = this.getDefaultSideMenu()
    
    this.setData({
      sideMenuList: defaultSideMenu,
      buttons: defaultButtons,
      isRapidMode: false,
      stepDistance: 1.0,
      // 应用信息
      appName: '虚拟键盘',
      appVersion: 'v1.0.0',
      logoPath: '/static/logo.png',
      // 简化的主题配置
      primaryColors: {
        background: 'linear-gradient(135deg, #323844 0%, #2c3e50 100%)',
        cardBackground: 'linear-gradient(145deg, rgba(255,255,255,0.15), rgba(255,255,255,0.08))',
        textPrimary: '#ffffff',
        textSecondary: '#bdc3c7',
        accent: '#3498db'
      },
      buttonColors: {
        green: '#27ae60',
        red: '#e74c3c', 
        blue: '#3498db',
        yellow: '#f1c40f',
        brown: '#8b4513',
        gray: '#7f8c8d',
        white: '#ecf0f1'
      },
      backgroundImage: ''
    })
    
    console.log('Default configuration loaded - buttons:', defaultButtons.length, 'menu items:', defaultSideMenu.length)
  },

  // 初始化API
  initializeApi() {
    motionControlApi.init()
    // 加载步进距离设置
    const defaults = configManager.getDefaults()
    const stepDistance = wx.getStorageSync('stepDistance') || defaults.stepDistance || 1.0
    this.setData({
      stepDistance: stepDistance
    })
  },

  // 切换侧边菜单
  toggleSideMenu() {
    this.setData({
      sideMenu: !this.data.sideMenu
    })
  },

  // 关闭侧边菜单
  closeSideMenu() {
    this.setData({
      sideMenu: false
    })
  },

  // 点击侧边菜单项
  clickSideMenu(e) {
    const index = e.currentTarget.dataset.index
    const menuItem = this.data.sideMenuList[index]
    
    this.setData({
      sideMenu: false
    })
    
    if (menuItem && menuItem.url) {
      wx.navigateTo({
        url: menuItem.url
      })
    } else {
      // 兼容旧版本
      if (index === 0) {
        wx.navigateTo({
          url: '../setting/setting'
        })
      }
      if (index === 1) {
        wx.navigateTo({
          url: '../about/about'
        })
      }
    }
  },

  // 点击侧边关闭按钮
  clickSideBtmBtn() {
    this.setData({
      sideMenu: false
    })
  },

  // 触摸开始
  touchStart(e) {
    const index = e.currentTarget.dataset.index
    const button = this.data.buttons[index]
    
    this.touchAndSend(index, '1')
    this.handleButtonAction(button, 'start')
    
    // 检查振动功能是否启用
    if (configManager.isFeatureEnabled('vibration')) {
      wx.vibrateShort({
        success: function () {
          console.log('vibrate success')
        }
      })
    }
  },

  // 触摸结束
  touchEnd(e) {
    const index = e.currentTarget.dataset.index
    const button = this.data.buttons[index]
    
    this.touchAndSend(index, '0')
    this.handleButtonAction(button, 'end')
  },

  /**
   * 处理按钮动作 - 使用配置系统
   */
  async handleButtonAction(button, eventType) {
    if (!button || !button.action) return

    try {
      // 检查API是否已初始化
      if (!motionControlApi.isInitialized) {
        console.log('API not initialized, showing notification for:', button.action);
        if (eventType === 'start') {
          wx.showToast({
            icon: 'error',
            title: '请先在设置中初始化',
            duration: 1500
          });
        }
        return;
      }

      // 获取动作配置
      const actionConfig = configManager.getActionConfig(button.action)
      
      if (actionConfig) {
        await this.executeConfiguredAction(actionConfig, eventType)
      } else {
        // 回退到默认处理
        await this.executeDefaultAction(button, eventType)
      }
    } catch (error) {
      console.error('Button action error:', error)
      // API未初始化的错误已经在前面处理过，不需要再显示错误
      if (error.message !== 'API not initialized') {
        wx.showToast({
          icon: 'error',
          title: '操作失败',
          duration: 1500
        })
      }
    }
  },

  /**
   * 检查是否为需要API的动作
   */
  isApiAction(action) {
    const apiActions = [
      'start', 'pause', 'continue', 'stop', 'workZero',
      'moveUp', 'moveDown', 'moveLeft', 'moveRight', 'zUp', 'zDown'
    ];
    return apiActions.includes(action);
  },

  /**
   * 显示演示动作（当API未初始化时）
   */
  showDemoAction(button) {
    const actionMessages = {
      'start': '演示：设备启动',
      'pause': '演示：设备暂停', 
      'continue': '演示：设备继续',
      'stop': '演示：设备停止',
      'workZero': '演示：设置工作零点',
      'moveUp': '演示：向上移动',
      'moveDown': '演示：向下移动',
      'moveLeft': '演示：向左移动',
      'moveRight': '演示：向右移动',
      'zUp': '演示：Z轴上升',
      'zDown': '演示：Z轴下降'
    };

    const message = actionMessages[button.action] || `演示：${button.text || button.action}`;
    
    wx.showToast({
      icon: 'none',
      title: message,
      duration: 1000
    });
  },

  /**
   * 执行配置的动作
   */
  async executeConfiguredAction(actionConfig, eventType) {
    const { type, method, successMessage, confirmRequired, confirmMessage, additionalActions } = actionConfig

    // 检查是否需要确认
    if (confirmRequired && eventType === 'start') {
      const confirmed = await this.showConfirmDialog(confirmMessage || '确定要执行此操作吗？')
      if (!confirmed) return
    }

    switch (type) {
      case 'api':
        if (eventType === 'start') {
          await motionControlApi[method]()
          if (successMessage) this.showSuccess(successMessage)
          
          // 执行附加动作
          if (additionalActions) {
            for (const additionalAction of additionalActions) {
              await motionControlApi[additionalAction]()
            }
          }
        }
        break

      case 'continuous':
        const { startMethod, stopMethod, stopParams } = actionConfig
        if (eventType === 'start') {
          await motionControlApi[startMethod](this.data.isRapidMode)
        } else if (eventType === 'end') {
          await motionControlApi[stopMethod](...(stopParams || []))
        }
        break

      case 'toggle':
        if (eventType === 'start') {
          const { property, value } = actionConfig
          this.setData({ [property]: value })
          if (successMessage) this.showSuccess(successMessage)
        }
        break

      case 'menu':
        if (eventType === 'start') {
          await this[method]()
        }
        break

      case 'custom':
        if (eventType === 'start') {
          await this.executeCustomAction(method, actionConfig)
        }
        break
    }
  },

  /**
   * 执行默认动作（回退兼容）
   */
  async executeDefaultAction(button, eventType) {
    // 如果API未初始化，显示提示
    if (!motionControlApi.isInitialized) {
      if (eventType === 'start') {
        wx.showToast({
          icon: 'error',
          title: '请先在设置中初始化',
          duration: 1500
        });
      }
      return;
    }

    // 这里保留原来的switch逻辑作为备用
    switch (button.action) {
      case 'start':
        if (eventType === 'start') {
          try {
            await motionControlApi.startMachine()
            this.showSuccess('设备启动')
          } catch (error) {
            console.error('Start machine error:', error)
          }
        }
        break
      case 'pause':
        if (eventType === 'start') {
          try {
            await motionControlApi.pauseMachine()
            this.showSuccess('设备暂停')
          } catch (error) {
            console.error('Pause machine error:', error)
          }
        }
        break
      case 'continue':
        if (eventType === 'start') {
          try {
            await motionControlApi.continueMachine()
            this.showSuccess('设备继续')
          } catch (error) {
            console.error('Continue machine error:', error)
          }
        }
        break
      case 'stop':
        if (eventType === 'start') {
          try {
            await motionControlApi.stopMachine()
            this.showSuccess('设备停止')
          } catch (error) {
            console.error('Stop machine error:', error)
          }
        }
        break
      case 'blow':
        if (eventType === 'start') {
          this.showSuccess('执行吹气功能')
        }
        break
      case 'follow':
        if (eventType === 'start') {
          this.showSuccess('执行跟随功能')
        }
        break
      case 'clearAlarm':
        if (eventType === 'start') {
          this.showSuccess('清除报警')
        }
        break
      case 'redLight':
        if (eventType === 'start') {
          this.showSuccess('红光定位')
        }
        break
      case 'breakpointPos':
        if (eventType === 'start') {
          this.showSuccess('断点定位')
        }
        break
      case 'backward':
        if (eventType === 'start') {
          this.showSuccess('回退功能')
        }
        break
      case 'forward':
        if (eventType === 'start') {
          this.showSuccess('前进功能')
        }
        break
      case 'laserPulse':
        if (eventType === 'start') {
          this.showSuccess('激光点射')
        }
        break
      case 'k1':
      case 'k2': 
      case 'k3':
      case 'k4':
        if (eventType === 'start') {
          this.showSuccess(`执行${button.text}功能`)
        }
        break
      case 'findEdge':
        if (eventType === 'start') {
          this.showSuccess('寻边功能')
        }
        break
      case 'quickSwap':
        if (eventType === 'start') {
          this.showSuccess('一键交换')
        }
        break
      case 'calibrate':
        if (eventType === 'start') {
          this.showSuccess('标定功能')
        }
        break
      case 'fn':
        if (eventType === 'start') {
          this.showSuccess('Fn功能')
        }
        break
      case 'dryRun':
        if (eventType === 'start') {
          this.showSuccess('空运行模式')
        }
        break
      case 'workZero':
        if (eventType === 'start') {
          try {
            await motionControlApi.backToOriginAll()
            this.showSuccess('设置工作零点')
          } catch (error) {
            console.error('Work zero error:', error)
          }
        }
        break
      case 'walkBorder':
        if (eventType === 'start') {
          this.showSuccess('走边框功能')
        }
        break
      case 'highSpeed':
        if (eventType === 'start') {
          this.setData({ isRapidMode: true })
          this.showSuccess('高速模式')
        }
        break
      case 'lowSpeed':
        if (eventType === 'start') {
          this.setData({ isRapidMode: false })
          this.showSuccess('低速模式')
        }
        break
      case 'step':
        if (eventType === 'start') {
          this.showStepMenu()
        }
        break
      // 方向控制
      case 'moveUp':
        if (eventType === 'start') {
          try {
            await motionControlApi.moveAxisUp(this.data.isRapidMode)
          } catch (error) {
            console.error('Move up error:', error)
          }
        } else if (eventType === 'end') {
          try {
            await motionControlApi.stopAxis(1)
          } catch (error) {
            console.error('Stop axis error:', error)
          }
        }
        break
      case 'moveDown':
        if (eventType === 'start') {
          try {
            await motionControlApi.moveAxisDown(this.data.isRapidMode)
          } catch (error) {
            console.error('Move down error:', error)
          }
        } else if (eventType === 'end') {
          try {
            await motionControlApi.stopAxis(1)
          } catch (error) {
            console.error('Stop axis error:', error)
          }
        }
        break
      case 'moveLeft':
        if (eventType === 'start') {
          try {
            await motionControlApi.moveAxisLeft(this.data.isRapidMode)
          } catch (error) {
            console.error('Move left error:', error)
          }
        } else if (eventType === 'end') {
          try {
            await motionControlApi.stopAxis(0)
          } catch (error) {
            console.error('Stop axis error:', error)
          }
        }
        break
      case 'moveRight':
        if (eventType === 'start') {
          try {
            await motionControlApi.moveAxisRight(this.data.isRapidMode)
          } catch (error) {
            console.error('Move right error:', error)
          }
        } else if (eventType === 'end') {
          try {
            await motionControlApi.stopAxis(0)
          } catch (error) {
            console.error('Stop axis error:', error)
          }
        }
        break
      case 'zUp':
        if (eventType === 'start') {
          try {
            await motionControlApi.moveZAxisUp(this.data.isRapidMode)
          } catch (error) {
            console.error('Z up error:', error)
          }
        } else if (eventType === 'end') {
          try {
            await motionControlApi.stopAxis(2)
          } catch (error) {
            console.error('Stop Z axis error:', error)
          }
        }
        break
      case 'zDown':
        if (eventType === 'start') {
          try {
            await motionControlApi.moveZAxisDown(this.data.isRapidMode)
          } catch (error) {
            console.error('Z down error:', error)
          }
        } else if (eventType === 'end') {
          try {
            await motionControlApi.stopAxis(2)
          } catch (error) {
            console.error('Stop Z axis error:', error)
          }
        }
        break
      default:
        if (eventType === 'start') {
          this.showSuccess(`执行: ${button.text || button.action}`)
        }
        break
    }
  },

  /**
   * 执行自定义动作
   */
  async executeCustomAction(method, actionConfig) {
    try {
      // 检查自定义功能是否启用
      if (!configManager.isFeatureEnabled('customActions')) {
        this.showSuccess('自定义功能未启用')
        return
      }

      // 执行自定义方法
      if (typeof this[method] === 'function') {
        await this[method](actionConfig)
      } else {
        console.warn(`Custom method ${method} not found`)
        this.showSuccess(actionConfig.successMessage || '执行自定义功能')
      }
    } catch (error) {
      console.error('Custom action error:', error)
    }
  },

  /**
   * 显示确认对话框
   */
  showConfirmDialog(message) {
    return new Promise((resolve) => {
      wx.showModal({
        title: '确认',
        content: message,
        showCancel: true,
        success: (res) => {
          resolve(res.confirm)
        },
        fail: () => {
          resolve(false)
        }
      })
    })
  },

  /**
   * 自定义动作示例
   */
  async customBlow(actionConfig) {
    // 这里可以实现吹气功能的自定义逻辑
    console.log('执行自定义吹气功能')
    this.showSuccess(actionConfig.successMessage || '吹气功能执行')
  },

  async customFollow(actionConfig) {
    // 这里可以实现跟随功能的自定义逻辑
    console.log('执行自定义跟随功能')
    this.showSuccess(actionConfig.successMessage || '跟随功能执行')
  },

  /**
   * 显示步进菜单
   */
  showStepMenu() {
    const { stepDistance } = this.data
    wx.showActionSheet({
      itemList: [
        `X轴左移 ${stepDistance}mm`,
        `X轴右移 ${stepDistance}mm`, 
        `Y轴上移 ${stepDistance}mm`,
        `Y轴下移 ${stepDistance}mm`,
        `Z轴上移 ${stepDistance}mm`,
        `Z轴下移 ${stepDistance}mm`,
        '设置步进距离'
      ],
      success: (res) => {
        this.handleStepAction(res.tapIndex)
      }
    })
  },

  /**
   * 处理步进动作
   */
  async handleStepAction(index) {
    const { stepDistance, isRapidMode } = this.data
    
    try {
      switch (index) {
        case 0: // X轴左移
          await motionControlApi.controlAxisInchMove(0, 0, stepDistance, isRapidMode)
          this.showSuccess(`X轴左移${stepDistance}mm`)
          break
        case 1: // X轴右移
          await motionControlApi.controlAxisInchMove(0, 1, stepDistance, isRapidMode)
          this.showSuccess(`X轴右移${stepDistance}mm`)
          break
        case 2: // Y轴上移
          await motionControlApi.controlAxisInchMove(1, 0, stepDistance, isRapidMode)
          this.showSuccess(`Y轴上移${stepDistance}mm`)
          break
        case 3: // Y轴下移
          await motionControlApi.controlAxisInchMove(1, 1, stepDistance, isRapidMode)
          this.showSuccess(`Y轴下移${stepDistance}mm`)
          break
        case 4: // Z轴上移
          await motionControlApi.controlAxisInchMove(2, 0, stepDistance, isRapidMode)
          this.showSuccess(`Z轴上移${stepDistance}mm`)
          break
        case 5: // Z轴下移
          await motionControlApi.controlAxisInchMove(2, 1, stepDistance, isRapidMode)
          this.showSuccess(`Z轴下移${stepDistance}mm`)
          break
        case 6: // 设置步进距离
          wx.navigateTo({
            url: '../setting/setting'
          })
          break
      }
    } catch (error) {
      console.error('Step action error:', error)
    }
  },

  /**
   * 显示成功提示
   */
  showSuccess(message) {
    wx.showToast({
      icon: 'success',
      title: message,
      duration: 1500
    })
  },

  /**
   * @description: 发送消息
   * @param {number} index 触摸开始或结束的键的索引
   * @param {string} newVal 键值 1是触摸开始 0是触摸结束或中断
   * @return {*}
   */
  touchAndSend(index, newVal) {
    console.log(`Button ${index} ${newVal === '1' ? 'pressed' : 'released'}`)
    
    let oldkeys = this.data.keys
    let newKeys = this.replaceStr(oldkeys, index, newVal)
    this.setData({
      keys: newKeys
    })
  },

  // 替换字符串指定位置的字符
  replaceStr(str, i, newVal) {
    let arr = str.split('')
    arr[i] = newVal
    return arr.join('')
  },

  // 轮询检查连接状态
  setTimer() {
    // 检查连接功能是否启用
    if (configManager.isFeatureEnabled('connectionCheck')) {
      this.checkServerConnection()
    }
    
    setTimeout(() => {
      this.setTimer()
    }, 3000)
  },

  /**
   * 检查服务器连接状态
   */
  async checkServerConnection() {
    try {
      const serverUrl = wx.getStorageSync('serverUrl')
      if (!serverUrl) {
        this.setData({ linkState: false })
        return
      }

      // 尝试发送一个简单的请求来检查连接
      const url = serverUrl.startsWith('http') ? serverUrl : `http://${serverUrl}`
      const apiConfig = configManager.getApiConfig()
      const healthEndpoint = apiConfig.endpoints?.health || '/api/health'
      
     
      wx.request({
        url: `${url}${healthEndpoint}`,
        method: 'GET',
        timeout: 3000,
        success: () => {
          this.setData({ linkState: true })
        },
        fail: () => {
          this.setData({ linkState: false })
        }
      })
    } catch (error) {
      this.setData({ linkState: false })
    }
  },

  // 切换键盘显示/隐藏
  toggleKeyboard() {
    this.setData({
      hideKeyboard: !this.data.hideKeyboard
    })
  },

  // 页面显示时重新初始化
  onShow() {
    this.initializeApi()
    this.loadConfiguration()
  }
})
